#include "bsp.h"
#include "spi.h"
#include "tty.h"

#define	D_IN	d_in()
#define	D_OUT	d_out()
#define	DATA	p0.b

#define	RS	p1.b6
#define	RW	p1.b5
#define	E	p1.b7

#define	REG	RS = 0
#define	CHAR	RS = 1
#define	WRITE	RW = 0
#define	READ	RW = 1
#define	DISABLE	E = 0
#define	ENABLE	E = 1

#define	TC_CS	p3.b4

#define	HEAT	p1.b1
#define	HEAT_ON	 0
#define	HEAT_OFF 1

#define UNITS	p5.b3
#define MODE	p5.b4

typedef enum {
  TempMode,
  TimeOnMode,
  TimeAtTempMode,
  DutyCycleMode,
  MaxMode
} Mode;

Mode mode = TempMode;
unsigned int mode_timeout = 0;
#define MODE_TIMEOUT 15000

/*****************************************************************************/

static volatile int hz = 0;
static volatile char heat = HEAT_OFF;
volatile unsigned long uptime_seconds = 0;

/* This is triggered by our line-monitor opto.  It's a pulse near the
   peak of each cycle.  */
void __attribute__((interrupt))
int1_handler ()
{
  trbocr.b = 0x01; /* trigger timer RB */
  hz ++;
  if (hz == 120)
    {
      hz = 0;
      uptime_seconds ++;
    }
}

/* This is intended to trigger right at zero crossings.  */
void __attribute__((interrupt))
timer_rb_handler ()
{
  trbic = 5;
  HEAT = heat;
}

/*****************************************************************************/

extern inline void
d_in ()
{
  prcr.b = 0xff; /* enable writes to pd0 */
  pd0.b = 0x00;
}
extern inline void
d_out ()
{
  prcr.b = 0xff; /* enable writes to pd0 */
  pd0.b = 0xff;
}

/* Our custom characters.  */
unsigned char cgram[] = {
  /* 0 = Copyright Symbol */
#define CHAR_COPYRIGHT	0
  0b01110,
  0b10001,
  0b10111,
  0b11001,
  0b10111,
  0b10001,
  0b01110,
  0b00000,

  /* 1 = "degrees F" */
#define CHAR_DEGF	1
  0b11000,
  0b11000,
  0b00111,
  0b00100,
  0b00111,
  0b00100,
  0b00100,
  0b00000,

  /* 2 = "degrees C" */
#define CHAR_DEGC	2
  0b11000,
  0b11000,
  0b00011,
  0b00100,
  0b00100,
  0b00100,
  0b00011,
  0b00000,

};

/* Read a byte from the LCD */
static int
read (int rs)
{
  int rv;
  READ;
  RS = rs;
  D_IN;
  ENABLE;
  rv = DATA;
  DISABLE;
  return rv;
}

/* Wait until the LCD is no longer busy */
static void
waitbusy ()
{
  while (1)
    {
      unsigned char status = read (REG);
      if ((status & 0x80) == 0)
	return;
    }
}

/* Write a byte out to the LCD */
static void
write (int rs, int d)
{
  waitbusy();
  WRITE;
  RS = rs;
  D_OUT;
  ENABLE;
  DATA = d;
  DISABLE;
}

static void
wstr (const char *s)
{
  while (*s)
    write (CHAR, *s++);
}

static void
wstrn (const char *s, int n)
{
  while (n)
    {
      write (CHAR, *s++);
      n--;
    }
}

#define TOP_ROW() write (REG, 128 + 0); /* ddram addr */
#define BOTTOM_ROW() write (REG, 192 + 0); /* ddram addr */

/*****************************************************************************/

static char buf[8];

/* Convert a number to a string.  Much less code space needed than
   calling siprintf.  Leading zeros are replaced with spaces.  */
char *
temp2string (int temp)
{
  int i;
  char sign = 0;

  if (temp < 0)
    {
      sign = 1;
      temp = -temp;
    };

  buf[0] = (temp/1000) + '0';
  buf[1] = ((temp/100)%10) + '0';
  buf[2] = ((temp/10)%10) + '0';
  buf[3] = (temp%10) + '0';
  for (i=0; i<3; i++)
    if (buf[i] == '0')
      buf[i] = ' ';
    else
      break;
  if (sign)
    buf[i-1] = '-';
  return buf;
}

static const char hex[] = "0123456789ABCDEF";

char *
temp2hex (unsigned int temp)
{
  buf[0] = hex[(temp >> 12) & 0x0f];
  buf[1] = hex[(temp >>  8) & 0x0f];
  buf[2] = hex[(temp >>  4) & 0x0f];
  buf[3] = hex[(temp      ) & 0x0f];
  return buf;
}

char *
time2string (long sec)
{
  int min = (sec/60) % 60;
  int hour = sec / (60*60);
  sec %= 60;
  buf[0] = hour/10 + '0';
  buf[1] = hour%10 + '0';
  buf[2] = ':';
  buf[3] = min/10 + '0';
  buf[4] = min%10 + '0';
  buf[5] = ':';
  buf[6] = sec/10 + '0';
  buf[7] = sec%10 + '0';
  return buf;
}

/*****************************************************************************/

/* Read one channel of the ADC.  Channel 8 is the temperature setting
   potentiometer.  */
int
adc (int channel)
{
  static int rv;
  if (channel < 8)
    {
      adcon0.adgsel0 = 0;
      adcon0.ch = channel;
    }
  else
    {
      adcon0.adgsel0 = 1;
      adcon0.ch = channel - 4;
    }
  adcon0.adst = 1;
  while (adcon0.adst)
    asm("");
  return ad;
}

/*****************************************************************************/

unsigned volatile int tics = 0;
unsigned volatile long tics_long = 0;

void __attribute__((interrupt))
timer_ra_handler ()
{
  tics ++;
  tics_long ++;
  traic = 5;
  if (mode_timeout)
    {
      mode_timeout --;
      if (mode_timeout == 0)
	mode = TempMode;
    }
}

static void __attribute__((constructor))
timer_A_initialize ()
{
  tracr.b = 0;
  traioc.b = 0;
  tramr.b = 0;
#ifdef CLOCK_125KHz
  trapre = 0;
  tra = 124;
#else
  trapre = (CLOCK_HZ / 100000) - 1;
  tra = 100-1;
#endif
  traic = 5;
  tracr.tstart = 1;
}

int
wait_ms (int ms)
{
  int now = tics;
  while ((int)(tics - now) < ms)
    asm("wait");
}

/*****************************************************************************/

main()
{
  int temp = 50;
  int i;
  int celsius = 0;
  int old_units = 0;
  int old_modekey = 0;
  int old_heat = HEAT_OFF;
  int status_tick = 0;

  int pot_idle_val = 0;
  int pot_idle_time = 0;
  int pot_idle = 0;

  int heat_on_time = 0;
  int heat_off_time = 0;
  int at_temp_time = 0;
  int last_heat_change_time = 0;

  pd1.b = 0xe2; /* LCD control signals, heat */
  pd3.b = 0x10; /* SPI */
  pd5.b = 0x00; /* Switches */
  DISABLE;

  TC_CS = 1;
  spi_reset ();

  tty_init ();

  /* The LCD needs time to initialize.  */
  wait_ms (30);

  tty_puts ("\nLaminator Monitor\n");

  /* 60 HZ line monitor interrupt.  */
  inten.b = 0x04;
  int1ic = 5;	/* falling edge */
  pmr.int1sel = 1;

  /* Timer RB is a 2.6ms one-shot.  */
  trbioc.b = 0;
  trbmr.b = 0x1a; /* f8, writable, one-shot */
  trbcr.b = 1; /* start */
  trbpre = 100-1;
  trbpr = ((CLOCK_HZ/800) * 0.0026) - 1;
  trbic = 5;

  /* ADC */
  adcon0.b = 0x80; /* f2, p0.0, repeat, off */
  adcon1.b = 0x28; /* f2, Vref, 10 bit */
  adcon2.b = 0x00; /* sample and hold */

  /* Initialize the LCD */
  write (REG, 0x38); /* 8-bit, 2-line, 5x8 */
  write (REG, 0x0f); /* display on, cursor on, cursor blink. */
  write (REG, 0x01);	/* clear display */
  wait_ms(2);
  write (REG, 0x06);	/* entry mode: ddram shift right */

  write (REG, 64 + 0); /* cgram address */
  for (i=0; i<sizeof (cgram); i++)
    write (CHAR, cgram[i]);

  TOP_ROW ();
  write (CHAR, CHAR_COPYRIGHT);
  wstr ("2009 DJ");
  BOTTOM_ROW ();
  wstr ("Delorie ");

  wait_ms (1000);

  while (1)
    {
      char *str;
      int dt;
      unsigned char tch, tcl;
      int units = UNITS;
      int modekey = MODE;

      /* See if the user pressed the C/F button.  */
      if (units != old_units)
	{
	  if (units)
	    celsius = ! celsius;
	  old_units = units;
	}

      /* See if the user pressed the Mode button.  */
      if (modekey != old_modekey)
	{
	  if (modekey)
	    {
	      mode ++;
	      if (mode == MaxMode)
		mode = TempMode;
	      mode_timeout = MODE_TIMEOUT;
	      //cprintf("mode: %d\n", mode);
	    }
	  old_modekey = modekey;
	}

      /* Read the potentiometer to determine the set temperature.  The
	 ADC range is 0..1023; we scale this to 0..512.  */
      dt = adc(8);

      /* keep the value from oscillating when it's "idle".  */
      if (pot_idle)
	{
	  /* If we're idle, watch for big changes.  */
	  if (dt < pot_idle_val - 10
	      || dt > pot_idle_val + 10)
	    {
	      pot_idle = 0;
	      pot_idle_val = dt;
	      pot_idle_time = tics;
	    }
	  /* But if there aren't any, re-use the old value.  */
	  dt = pot_idle_val;
	}
      else
	{
	  /* If we're active, watch for little changes - set this
	     small enough to detect actual motion, but big enough to
	     ignore ADC noise.  */
	  if (dt < pot_idle_val - 3
	      || dt > pot_idle_val + 3)
	    {
	      pot_idle = 0;
	      pot_idle_val = dt;
	      pot_idle_time = tics;
	    }
	  /* If we haven't seen any activity for 2 seconds, go idle.  */
	  if ((tics - pot_idle_time) > 2000)
	    {
	      pot_idle = 1;
	      pot_idle_val = dt;
	    }
	  at_temp_time = 0;
	}

      if (celsius)
	dt = (dt+512) / 8;
      else
	dt = (dt+512) * 9/(5*8) + 32;

      /* Read the thermocouple temperature.  */
      TC_CS = 0;
      tch = spi_read ();
      tcl = spi_read ();
      TC_CS = 1;

      temp = tch * 256 + tcl;
      temp >>= 3;
      /* now 0.25C per LSB */
      if (celsius)
	temp = temp >> 2;
      else
	temp = temp * 9 / (5*4) + 32;

      if (temp < dt)
	heat = HEAT_ON;
      else if (temp > dt)
	heat = HEAT_OFF;

      if (heat != old_heat)
	{
	  if (heat == HEAT_ON)
	    {
	      heat_off_time = uptime_seconds - last_heat_change_time;
	      //cprintf("heat_off_time %d\n", heat_off_time);
	    }
	  else
	    {
	      heat_on_time = uptime_seconds - last_heat_change_time;
	      //cprintf("heat_on_time %d\n", heat_on_time);
	    }
	  if (at_temp_time == 0)
	    {
	      at_temp_time = uptime_seconds;
	      //cprintf("at_temp_time %d\n", at_temp_time);
	    }
	  last_heat_change_time = uptime_seconds;
	  //cprintf("last_heat_change_time %d\n", last_heat_change_time);
	  old_heat = heat;
	}

      switch (mode)
	{
	case TempMode:
	  TOP_ROW();
	  wstr ("Set");
	  str = temp2string(dt);
	  wstrn (str, 4);
	  write (CHAR, celsius ? CHAR_DEGC : CHAR_DEGF);


	  BOTTOM_ROW();
	  wstr ("Lam");
	  str = temp2string(temp);
	  wstrn (str, 4);
	  write (CHAR, celsius ? CHAR_DEGC : CHAR_DEGF);
	  break;

	case TimeOnMode:
	  TOP_ROW();
	  wstr ("Time On ");
	  BOTTOM_ROW();
	  wstrn (time2string(uptime_seconds), 8);
	  break;

	case TimeAtTempMode:
	  TOP_ROW();
	  wstr ("Time At ");
	  BOTTOM_ROW();
	  if (at_temp_time)
	    wstrn (time2string(uptime_seconds - at_temp_time), 8);
	  else
	    wstr("--:--:--");
	  break;

	case DutyCycleMode:
	  TOP_ROW();
	  wstr ("Duty Cyc");
	  BOTTOM_ROW();
	  str = temp2string (heat_on_time);
	  wstrn (str+1, 3);
	  write (CHAR, 'H');
	  str = temp2string (heat_off_time);
	  wstrn (str+1, 3);
	  write (CHAR, 'C');
	  break;

	}

      if (++status_tick == 4)
	{
	  //cprintf("%d %d %d %c %d\n", hz, dt, temp, celsius ? 'C' : 'F', heat);
	  status_tick = 0;
	}

      /* The MAX6675 is spec'd at 220mS max conversion time.  */
      wait_ms(250);
    }
}
